#!/bin/bash

cdir=`pwd`
prog=`basename $0`
force=1
animate=1
#-------------------------------------------------------------------------------------------
# If you want newlines, you have to echo "$USAGE" and not echo $USAGE
USAGE="\
Usage: ${prog} -f <force> k1 k2 k3 k4 x10 x20 x30 v10 v20 v30 STEPS t0 tf
Other Options:
 -n Do not animate trajectory
Available forces (value of <force>):
1: ax=-k2 vx    ay=-k2 vy    az=-k2 vz - k1     Free fall
2: ax= k2+k1 vy ay= k3-k1 vx az=k4              E+B field:  qB/m=k1 z qE/m=k2 x+k3 y+k4 z
3: ax= (k1+k2*x3p)vy ay= -(k1+k2*x3p) * vx az=0 where x3p=(x3>0)?x3:0 B field with variable z-component. Not physical!
4: ax=vy(k1+k2 z)+vz k2 y ay= -vx (k1+k2 z) az= -vx k2 y qBy/m= -k2 y qBz/m = (k1+k2 z)k
5: a1=v2(k1+k2 x2) a2=-v1(k1+k2 x2) a3 = v1 k3 x3  qBy/m= k3 x3 qBz/m =k1+k2 x2
6: a1=-k1 x     ay=-k2 y     az=-k3 z           Simple harmonic oscillator
"
#-------------------------------------------------------------------------------------------
set -u # Undo: set +u    Treat unset variables and parameters other than the special parameters "@" and "*" as an error 
while getopts :hf:n opt_char; do
 case $opt_char in
  x)  animate=0                                                 ;;
  f)  force=$OPTARG                                             ;;
  h)  echo "$USAGE"                                     ;exit 1 ;;
 \?)  echo "$OPTARG is not a valid option";echo "$USAGE";exit 1 ;;
 esac
done
shift $((OPTIND-1))
if(( $# != 13 ));then echo "$USAGE";exit;fi  # If arguments are necessary, uncomment this line
#-------------------------------------------------------------------------------------------
acalc     () { awk  -v OFMT="%.17g" "BEGIN{print $* }"      ; } # Works as function: pi2=`acalc "2 * $pi"`
extension () { echo     "$*" | awk -F'.'   'NF>1{print $NF}'; } # extension file.tar.gz = gz
filename  () { basename "$1"              ".${1##*.}"       ; } # filename  file.tar.gz = file.tar
#-------------------------------------------------------------------------------------------


k1=${1}
k2=${2}
k3=${3}
k4=${4}
x10=${5}
x20=${6}
x30=${7}
v10=${8}
v20=${9}
v30=${10}
STEPS=${11}
t0=${12}
tf=${13}

forcecode=(g B Bz Bzy Byz hoc)

if (( force > ${#forcecode[@]} ));then
 echo "${prog}: Not valid code for type of force"
 echo "$USAGE"
 exit 1
fi

if [[ -f rk3           ]];then rm rk3          ;fi


(( force-- ))
gfortran -O2 rk3_${forcecode[$force]}.f90 rk3.f90 rksuite/rksuite.f -o rk3
./rk3 <<EOF
$k1 $k2 $k3 $k4
$STEPS $t0 $tf $x10 $x20 $x30 $v10 $v20 $v30
EOF

gnuplot -persist -d <<EOF
set term qt  1
plot "rk3.dat" using 1:2 with lines title "x1(t)"
set term qt  2
plot "rk3.dat" using 1:3 with lines title "x2(t)"
set term qt  3
plot "rk3.dat" using 1:4 with lines title "x3(t)"
set term qt  4
plot "rk3.dat" using 1:5 with lines title "v1(t)"
set term qt  5
plot "rk3.dat" using 1:6 with lines title "v2(t)"
set term qt  6
plot "rk3.dat" using 1:7 with lines title "v3(t)"
set term qt  7
plot "rk3.dat" using 1:8 with lines title "E(t)"
set term qt  8
set title "trajectory"
set xlabel "x"
set ylabel "y"
set zlabel "z"
splot "rk3.dat" u 2:3:4 w l notit
EOF


if(( animate == 1));then
 skip=$(awk -v s=$STEPS 'BEGIN{s=int(s/100);if(s<1){s=1};print s}')
 ./rk3_animate.sh -d $skip -r
fi
#  ---------------------------------------------------------------------
#  Copyright by Konstantinos N. Anagnostopoulos (2004-2021)
#  Physics Dept., National Technical University,
#  konstant@mail.ntua.gr, www.physics.ntua.gr/~konstant
#  
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, version 3 of the License.
#  
#  This program is distributed in the hope that it will be useful, but
#  WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  General Public License for more details.
#  
#  You should have received a copy of the GNU General Public Liense along
#  with this program.  If not, see <http://www.gnu.org/licenses/>.
#  -----------------------------------------------------------------------
