#!/bin/bash

cdir=`pwd`
prog=`basename $0`
#-------------------------------------------------------------------------------------------
force=1
animate=1
#-------------------------------------------------------------------------------------------
# If you want newlines, you have to echo "$USAGE" and not echo $USAGE
USAGE="\
Usage: ${prog} -f <force> k1 k2 k3 k4 x10 x20 x30 v10 v20 v30 STEPS t0 tf
Other Options:
 -n Do not animate trajectory
Available forces (value of <force>):
1: ax= k2+k1 vy ay= k3-k1 vx az=k4              E+B field:  qB/m=k1 z qE/m=k2 x+k3 y+k4 z
2: B_1 = k1 (3 x1 x3)/r^5 B_2 = k1 (3 x2 x3)/r^5 B_3 = k1[(3 x3 x3)/r^5-1/r^3] Magnetic dipole
"
#-------------------------------------------------------------------------------------------
set -u # Undo: set +u    Treat unset variables and parameters other than the special parameters "@" and "*" as an error 
while getopts :hnf: opt_char; do
 case $opt_char in
  f)  force=$OPTARG                                             ;;
  n)  animate=0                                                 ;;
  h)  echo "$USAGE"                                     ;exit 1 ;;
 \?)  echo "$OPTARG is not a valid option";echo "$USAGE";exit 1 ;;
 esac
done
shift $((OPTIND-1))
if(( $# != 13 ));then echo "$USAGE";exit;fi  # If arguments are necessary, uncomment this line
#-------------------------------------------------------------------------------------------
acalc     () { awk  -v OFMT="%.17g" "BEGIN{print $* }"      ; } # Works as function: pi2=`acalc "2 * $pi"`
extension () { echo     "$*" | awk -F'.'   'NF>1{print $NF}'; } # extension file.tar.gz = gz
filename  () { basename "$1"              ".${1##*.}"       ; } # filename  file.tar.gz = file.tar
#-------------------------------------------------------------------------------------------

 k1=${1}
 k2=${2}
 k3=${3}
 k4=${4}
x10=${5}
x20=${6}
x30=${7}
v10=${8}
v20=${9}
v30=${10}
STEPS=${11}
 t0=${12}
 tf=${13}

# Check that velocities are well defined:
errv=$(awk -v v1=$v10 -v v2=$v20 -v v3=$v30 'BEGIN{vsq=v1*v1+v2*v2+v3*v3;if(vsq>=1.0){err=1}else{err=0};print err;}')

if(( errv == 1 ));then
 echo "${prog}: Sorry, light/superluminal particle not allowed."
 echo "         v1=$v10  v2=$v20  v3=$v30                      "
 exit 1
fi

forcecode=(B Bd)

if (( force > ${#forcecode[@]} ));then
 echo "${prog}: Not valid code for type of force"
 echo "$USAGE"
 exit 1
fi

if [[ -f sr           ]];then rm sr          ;fi


(( force-- ))
gfortran -O2 sr.f90 sr_${forcecode[$force]}.f90 rksuite/rksuite.f -o sr
./sr <<EOF
$k1 $k2 $k3 $k4
$STEPS $t0 $tf $x10 $x20 $x30 $v10 $v20 $v30
EOF


gnuplot -persist -d <<EOF
set term qt 1
plot "sr.dat" using 1:2  with lines title "x1(t)"
set term qt 2
plot "sr.dat" using 1:3  with lines title "x2(t)"
set term qt 3
plot "sr.dat" using 1:4  with lines title "x3(t)"
set term qt 4
plot "sr.dat" using 1:5  with lines title "v1(t)"
set term qt 5
plot "sr.dat" using 1:6  with lines title "v2(t)"
set term qt 6
plot "sr.dat" using 1:7  with lines title "v3(t)"
set term qt 7
plot "sr.dat" using 1:8  with lines title "E(t)"
set term qt 8
plot "sr.dat" using 1:9  with lines title "p1(t)"
set term qt 9
plot "sr.dat" using 1:10 with lines title "p2(t)"
set term qt 10
plot "sr.dat" using 1:11 with lines title "p3(t)"
set term qt 11
set title "trajectory"
set xlabel "x"
set ylabel "y"
set zlabel "z"
splot "sr.dat" u 2:3:4 w l notit
EOF

if(( animate == 1));then
 skip=$(awk -v s=$STEPS 'BEGIN{s=int(s/100);if(s<1){s=1};print s}')
 ./sr_animate.sh -d $skip -r
fi
